// Cloth Physics Demonstration
// Matt Pierce
// http://www.pie2k.com
// mpierce@pie2k.com

// This version is written for Borland
// Turbo C++ 3.0.  I have no idea if this
// will work anywhere else, but the only
// DOS specific stuff is in the display
// code.

// I'm pretty sure I'm using only standard
// C and C++ for the actual algorithm, and
// I'll try to flag anything non-standard.

///////////////////////////
// Included files
// Standard Stuff
#include <stdio.h>
#include <stdlib.h>
// Needed for sqrt() sin() and cos()
#include <math.h>
// Needed for graphics library
#include <mem.h>
// Used for getch()
#include <conio.h>
// A quick and simple graphics library
// for DOS.  Very barebones.
#include "graphics.h"


///////////////////////////
// Defines

// I used defines instead of constants
// because WIDTH*2 gets converted to
// 44*2 which the compiler changes to
// 88, saving me a multiply. :)
#define WIDTH 44
#define HEIGHT 40
#define ZOOM 0.40

///////////////////////////
// Macros

// HANDLE_SPRING does the actual spring
// physics.  You give it two points, a
// spring constant and default length
// and it'll create a block of code to do
// the math for you.  I did this to save on
// function call overhead.  It's a slow
// algorithm, so I'll cheat a little.
#define HANDLE_SPRING(a, b, k, l) d=0.5*k*(l - dist(a,b));\
			if (d>10*k) d=10*k;\
			fx=(b.px-a.px);\
			fy=(b.py-a.py);\
			fz=(b.pz-a.pz);\
			ratio=my_sqrt(fx*fx + fy*fy + fz*fz);\
			if (ratio!=0) ratio = d/ratio;\
			fx *= ratio;\
			fy *= ratio;\
			fz *= ratio;\
			a.fx -= fx; b.fx += fx;\
			a.fy -= fy; b.fy += fy;\
			a.fz -= fz; b.fz += fz

// SClothNode stores a vertex in
// the cloth.
struct SClothNode {
	float px, py, pz;		// position
	float vx, vy, vz;		// velocity
	float fx, fy, fz;};	// net forces acting on it

// SSphere
// a sphere to let the cloth drape across.
struct SSphere{
	float x,y,z,r;};

// Virtual screen
// (used for DOS graphics)
uchar virt[64000];

// sqrt_tab
// this is used to speed up
// calculations with square root.
// Why calculate something when you
// can look it up?
float *sqrt_tab;

void anim();				// Our main animation loop
void init_sqrt_tab();	// initialization for sqrt_tab
float my_sqrt(float i); // my own sqrt function, uses
								// sqrt_tab

// An inline function to get the distance between two SClothNodes.
// Done inline to speed things up.  It's basically a function for
// the sake of cleaning up the code.
inline float dist (const SClothNode &a, const SClothNode &b)
{
	// uses pythagorean formula to get the distance between two nodes
	return my_sqrt((a.px-b.px)*(a.px-b.px) + (a.py-b.py)*(a.py-b.py) + (a.pz-b.pz)*(a.pz-b.pz));
}

//////////////////
// Main function... if you don't
// know what this is, you shouldn't
// be reading this tutorial. :P
void main()
{
	// DOS function, go into 320x200x8 graphics mode
	setmode(0x13);
	// DOS funciton, set greyscale pallette
	greyscale();
	// initialize the square root table.
	init_sqrt_tab();
	// enter animation loop until user quits
	anim();
	// DOS function, go back to text mode
	setmode(0x03);
}

/////////////////
// Anim function
// This is the function where everything
// happens.  There's no explanation other
// than that.  You'll see.
void anim()
{
	// coordinates and loop counters
	int x,y,a,b,p,q;
	// forces
	float fx, fy, fz, ratio, d;
	// animation timer, mass, and something
	// I can't remember.
	float k, m, scalar;

	// spring constants.  For basic springs,
	// diagonal springs, and long springs.
	float ka = 150.0;
	float kb = 150.0;
	float kc = 100.0;
	// mass
	m = 5.0;

	// Create a sphere object
	SSphere my_sphere;

	// Create the cloth
	SClothNode *node = new SClothNode[HEIGHT*WIDTH];

	// Initialize the cloth
	// make it like a bilboard, and then
	// it'll hang from its corners (like
	// a sheet on a clothesline)
	for (x=0; x<WIDTH; x++)
		for (y=0; y<HEIGHT; y++)
		{
			node[y*WIDTH+x].px = 10*(x-(WIDTH/2.0));
			node[y*WIDTH+x].py = 10*(y-(HEIGHT/2.0));
			node[y*WIDTH+x].pz = 0;
			node[y*WIDTH+x].vx = 0;
			node[y*WIDTH+x].vy = 0;
			node[y*WIDTH+x].vz = 0;
		}

	// Create a sphere to map against
	my_sphere.x = 0;
	my_sphere.y = 0;
	my_sphere.z = 0;
	my_sphere.r = 150.0;

	// Enter Loop
	while (!kbhit())
	{
		// Incriment animation timer
		k += 0.02;

		// Clear Screen
		clear(virt, 0);

		// Draw Sphere (what a cop out)
		circle(ZOOM*my_sphere.x+160,
				 100-ZOOM*my_sphere.y,
				 ZOOM*my_sphere.r,196,virt);

		// set default forces for each node.
		// wind in the x direction... a little in the z direction,
		// and gravity in the y direction.
		y=WIDTH*HEIGHT;
		fx = 0 + 8.0*sin(k/10.0)*sin(k/10.0)*sin(k/10.0)*(0.7+sin(k/3.0));
		for (x=0;x<y;x++)
		{
			node[x].fx = fx;
			node[x].fy = -0.98*3.5;
			node[x].fz = 0.05;
		}

		// Handle Horizontal Lateral Springs
		for (x=0; x<WIDTH-1; x++)
		{
			for (y=0; y<HEIGHT; y++)
			{
				a = y*WIDTH + x;
				b = a + 1;
				HANDLE_SPRING(node[a], node[b], ka, 10.0);
			}
		}

		// Handle Vertical Lateral Springs
		for (x=0; x<WIDTH; x++)
		{
			for (y=0; y<HEIGHT-1; y++)
			{
				a = y*WIDTH + x;
				b = a + WIDTH;
				HANDLE_SPRING(node[a], node[b], ka, 10.0);
			}
		}

		// Handle Diagonal Springs
		for (x=0; x<WIDTH-1; x++)
		{
			for (y=0; y<HEIGHT-1; y++)
			{
				a = y*WIDTH+x;
				b = a+WIDTH+1;
				HANDLE_SPRING(node[a], node[b], kb, 14.14);
				a += 1;
				b -= 1;
				HANDLE_SPRING(node[a], node[b], kb, 14.14);

			}
		}

		// Handle Horizontal Long Springs
		for (x=0; x<WIDTH-2; x++)
		{
			for (y=0; y<HEIGHT; y++)
			{
				a = y*WIDTH + x;
				b = a + 2;
				HANDLE_SPRING(node[a], node[b], kc, 20.0);
			}
		}

		// Handle Vertical Lateral Springs
		for (x=0; x<WIDTH; x++)
		{
			for (y=0; y<HEIGHT-2; y++)
			{
				a = y*WIDTH + x;
				b = a + WIDTH*2;
				HANDLE_SPRING(node[a], node[b], kc, 20.0);
			}
		}

		// Move Cloth
		// Basically, take the forces being exerted
		// on each point, and then alter the velocity
		// which alters the position.
		y = WIDTH*HEIGHT;
		for (x=0;x<y;x++)
		{
			// Dampen the movement.
			// As in any physics simulation, you have
			// to have some energy loss.
			node[x].vx *= 0.99;
			node[x].vy *= 0.99;
			node[x].vz *= 0.99;
			// Alter the velocity based on force
			node[x].vx += node[x].fx/m;
			node[x].vy += node[x].fy/m;
			node[x].vz += node[x].fz/m;
			// Alter the positions based on velocity
			node[x].px += node[x].vx/50.0;
			node[x].py += node[x].vy/50.0;
			node[x].pz += node[x].vz/50.0;
		}

		// Contact Map With Sphere and each point
		my_sphere.x = 300.0*cos(k/5.0);
		my_sphere.y = 50.0*sin(k/1.0);
		my_sphere.z = 10.0*sin(k*3.7);
		y = WIDTH*HEIGHT;
		for (x=0;x<y;x++)
		{
			// find the difference between the
			// point and the center of the sphere
			fx = node[x].px - my_sphere.x;
			fy = node[x].py - my_sphere.y;
			fz = node[x].pz - my_sphere.z;
			// avoid a devide by zero error
			if ((fx==0) && (fy==0) && (fz==0)) fy = 0.1;
			d = fx*fx + fy*fy + fz*fz;
			// if the point is inside the sphere.
			if (d<(my_sphere.r*my_sphere.r))
			{
				// get the correct distance.
				d = my_sqrt(d);
				if (d==0) d=0.1;
				// normalize the difference vector
				fx /= d;
				fy /= d;
				fz /= d;
				// push out in that direction to the
				// surface of the sphere
				node[x].px = my_sphere.x + my_sphere.r*fx;
				node[x].py = my_sphere.y + my_sphere.r*fy;
				node[x].pz = my_sphere.z + my_sphere.r*fz;
				// I don't remember why this is called scalar.
				// Basically it quashes any velocity into the sphere,
				// leaving only velocity along the surface of the
				// sphere.
				scalar = (-fx*node[x].vx - fy*node[x].vy - fz*node[x].vz)/
							(fx*fx + fy*fy + fz*fz);
				node[x].vx += fx*scalar;
				node[x].vy += fy*scalar;
				node[x].vz += fz*scalar;

			}
		}

		// Clamp "Attached" Points
		b = HEIGHT*WIDTH-1;
		a = b-WIDTH+1;
		// move 'em up and down, like attached to a
		// clothesline swinging in the breeze.
		node[a].px = 10*(-(WIDTH/2));
		node[a].py = 10*(HEIGHT/2) + 25*cos(k*1.5);
		node[a].pz = 0;
		node[b].px = 10*(WIDTH/2);
		node[b].py = node[a].py;
		node[b].pz = 0;

		// Draw Cloth
		// This is very messy, as I'm very lazy.
		for (x=0;x<WIDTH-1;x++)
		{
			for (y=0;y<HEIGHT-1;y++)
			{
				a = ZOOM*node[y*WIDTH+x].px+160;
				b = 100-ZOOM*node[y*WIDTH+x].py;
				p = ZOOM*node[y*WIDTH+x+1].px+160;
				q = 100-ZOOM*node[y*WIDTH+x+1].py;
				rline(a,b,p,q,250,virt);
				p = ZOOM*node[y*WIDTH+x+WIDTH].px+160;
				q = 100-ZOOM*node[y*WIDTH+x+WIDTH].py;
				rline(a,b,p,q,250,virt);
				pput(a,b,250,virt);

			}
		}

		for (x=0;x<WIDTH-1;x++)
		{
			y = HEIGHT-1;
			a = ZOOM*node[y*WIDTH+x].px+160;
			b = 100-ZOOM*node[y*WIDTH+x].py;
			p = ZOOM*node[y*WIDTH+x+1].px+160;
			q = 100-ZOOM*node[y*WIDTH+x+1].py;
			rline(a,b,p,q,250,virt);
			pput(a,b,250,virt);
		}

		for (y=0;y<HEIGHT-1;y++)
		{
			x = WIDTH-1;
			a = ZOOM*node[y*WIDTH+x].px+160;
			b = 100-ZOOM*node[y*WIDTH+x].py;
			p = ZOOM*node[y*WIDTH+x+WIDTH].px+160;
			q = 100-ZOOM*node[y*WIDTH+x+WIDTH].py;
			rline(a,b,p,q,250,virt);
			pput(a,b,250,virt);
		}
		pput(p,q,250,virt);


		flip(virt);
	}
	getch();
}

// init_sqrt_tab
// Initializes my square root table.
void init_sqrt_tab()
{
	float i;
	sqrt_tab = new float[16000];
	for (int x=0;x<16000;x++)
	{
		i = x/10.0;
		sqrt_tab[x] = sqrt(i);
	}
}

// my_sqrt
// the square root function that
// uses the table.
float my_sqrt(float i)
{
	if(i>=1600){
		return sqrt(i);
	} else return sqrt_tab[(int)(i*10)];
}


