/*
	3D Math operations for Turbo C++ 3.0
	Written/converted by Matt Pierce

	If you have anything to contribute to this,
	please feel free to make changes and post them
	wherever. I didn't come up with any of these
	algorithms, they're just implementations of
	geometric/algebraec math principles which
	have been around for hundreds of years.

	The only thing that separates this
	implementation from others I've seen is that
	I'm overloading a lot of operators in an
	attempt at making the math easier to use.

	So basically, a little credit for the
	original implementation would be nice,
	but this code is most definitely
	public domain.

	If you need a primer on how any of this
	math works, look up Vectors and
	Matrices on:
	http://www.euclideanspace.com
	in the 3DMaths section
*/

#include <math.h>

#ifndef _3DMATH_H
#define _3DMATH_H

struct SVect3d {
	float x, y, z;
};
#define MacroDotProd(a, b) a.x*b.x + a.y*b.y +a.z*b.z
#define MacroLength(a) sqrt(a.x*a.x + a.y*a.y + a.z*a.z)
#define MacroTransform(v, m, r) r.x = v.x*m[0]+v.y*m[1]+v.z*m[2]+m[3];\
r.y = v.x*m[4]+v.y*m[5]+v.z*m[6]+m[7];\
r.z = v.x*m[8]+v.y*m[9]+v.z*m[10]+m[11];

float GTM[16]; //Global Temporary Matrix

inline SVect3d operator-(const SVect3d &a, const SVect3d &b)
{
	SVect3d t;
	t.x = a.x - b.x;
	t.y = a.y - b.y;
	t.z = a.z - b.z;
	return t;
}

inline SVect3d operator+(const SVect3d a, const SVect3d b)
{
	SVect3d t;
	t.x = a.x + b.x;
	t.y = a.y + b.y;
	t.z = a.z + b.z;
	return t;
}

inline SVect3d operator-(const SVect3d &v)
{
	SVect3d t;
	t.x = -v.x;
	t.y = -v.y;
	t.z = -v.z;
	return t;
}

inline SVect3d operator*(const SVect3d &a, const float &b)
{
	SVect3d t;
	t.x = a.x*b;
	t.y = a.y*b;
	t.z = a.z*b;
	return t;
}

inline SVect3d operator*(const float &a, const SVect3d &b)
{
	return b*a;
}

inline SVect3d operator/(const SVect3d &a, const float &b)
{
	SVect3d t;
	t.x = a.x/b;
	t.y = a.y/b;
	t.z = a.z/b;
	return t;
}

inline float Length(const SVect3d &v)
{
	return MacroLength(v);
}

inline SVect3d Normalize(const SVect3d &v)
{
	float l = MacroLength(v);
	if (l != 0.0f) return v/l;
	else return v;
}

inline float DotProd(const SVect3d &a, const SVect3d & b)
{
	return MacroDotProd(a,b);
}

inline float operator*(const SVect3d &a, const SVect3d & b)
{
	return MacroDotProd(a,b);
}

inline SVect3d CrossProd(const SVect3d &a, const SVect3d &b)
{
	SVect3d t;
	t.x = a.y*b.z - a.z*b.y;
	t.y = a.z*b.x - a.x*b.z;
	t.z = a.x*b.y - a.y*b.x;
	return t;
}

inline SVect3d operator^(const SVect3d &a, const SVect3d &b)
{
	SVect3d t;
	t.x = a.y*b.z - a.z*b.y;
	t.y = a.z*b.x - a.x*b.z;
	t.z = a.x*b.y - a.y*b.x;
	return t;
}

// Our matrices will be: float[16]
// And we'll pass them using: float *
void LoadIdentity(float far *mat)
{
	mat[0]  = 1.0f;
	mat[1]  = 0.0f;
	mat[2]  = 0.0f;
	mat[3]  = 0.0f;

	mat[4]  = 0.0f;
	mat[5]  = 1.0f;
	mat[6]  = 0.0f;
	mat[7]  = 0.0f;

	mat[8]  = 0.0f;
	mat[9]  = 0.0f;
	mat[10]  = 1.0f;
	mat[11]  = 0.0f;

	mat[12]  = 0.0f;
	mat[13]  = 0.0f;
	mat[14]  = 0.0f;
	mat[15]  = 1.0f;
}

void Concatenate(float far *m1, float far *m2, float far *result)
{
	unsigned int i, j, j1, j2, j3;
	float temp[16];
	for (i=0; i<4; i++)
	{
		j = 4*i;
		j1 = j+1;
		j2 = j+2;
		j3 = j+3;
		GTM[j ] = m2[j]*m1[0] + m2[j1]*m1[4] + m2[j2]*m1[8 ] + m2[j3]*m1[12];
		GTM[j1] = m2[j]*m1[1] + m2[j1]*m1[5] + m2[j2]*m1[9 ] + m2[j3]*m1[13];
		GTM[j2] = m2[j]*m1[2] + m2[j1]*m1[6] + m2[j2]*m1[10] + m2[j3]*m1[14];
		GTM[j3] = m2[j]*m1[3] + m2[j1]*m1[7] + m2[j2]*m1[11] + m2[j3]*m1[15];
	}
	for (i=0; i<16; i++) result[i] = GTM[i];
}

void Rotate(float angle, const float &x, const float &y, const float &z, float far *mat)
{
	float temp[16];

	float c = cos(angle);
	float s = sin(angle);
	float v = 1.0f - c;

	LoadIdentity(temp);

	temp[0] = x*x*v + c;
	temp[1] = x*y*v - z*s;
	temp[2] = x*z*v + y*s;
	temp[4] = x*y*v + z*s;
	temp[5] = y*y*v + c;
	temp[6] = y*z*v - x*s;
	temp[8] = x*z*v - y*s;
	temp[9] = y*z*v + x*s;
	temp[10]= z*z*v + c;

	Concatenate(mat, temp, mat);
}

void Scale(float x, float y, float z, float far *mat)
{
	float temp[16];
	int i;
	for (i=0;i<16;i++) temp[i] = 0.0f;

	temp[0] = x;
	temp[5] = y;
	temp[10]= z;
	temp[15]= 1.0f;
	Concatenate(mat, temp, mat);
}

void Translate(float x, float y, float z, float far *mat)
{
	float temp[16];
	int i;
	for (i=0; i<16; i++) temp[i] = 0;
	temp[0] = temp[5] = temp[10] = temp[15] = 1.0f;

	temp[3] = x;
	temp[7] = y;
	temp[11]= z;

	Concatenate(mat, temp, mat);
}

SVect3d Transform(const SVect3d &v, float far *mat)
{
	SVect3d t;

	t.x = v.x*mat[0] + v.y*mat[1] + v.z*mat[2] + mat[3];
	t.y = v.x*mat[4] + v.y*mat[5] + v.z*mat[6] + mat[7];
	t.z = v.x*mat[8] + v.y*mat[9] + v.z*mat[10]+ mat[11];

	return t;
}

void Invert(float far *mat, float far *imat)
{
	float tmp[12];
	float src[16];
	float det;

	for (int i=0; i<4; i++)
	{
		src[i] = mat[i*4];
		src[i+4] = mat[i*4+1];
		src[i+8] = mat[i*4+2];
		src[i+12]= mat[i*4+3];
	}

	tmp[0] = src[10]*src[15];
	tmp[1] = src[11]*src[14];
	tmp[2] = src[9]*src[15];
	tmp[3] = src[11]*src[13];
	tmp[4] = src[9]*src[14];
	tmp[5] = src[10]*src[13];
	tmp[6] = src[8]*src[15];
	tmp[7] = src[11]*src[12];
	tmp[8] = src[8]*src[14];
	tmp[9] = src[10]*src[12];
	tmp[10] = src[8]*src[13];
	tmp[11] = src[9]*src[12];

	imat[0] =  tmp[0]*src[5] + tmp[3]*src[6] + tmp[4]*src[7];
	imat[0] -= tmp[1]*src[5] + tmp[2]*src[6] + tmp[5]*src[7];
	imat[1] =  tmp[1]*src[4] + tmp[6]*src[6] + tmp[9]*src[7];
	imat[1] -= tmp[0]*src[4] + tmp[7]*src[6] + tmp[8]*src[7];
	imat[2] =  tmp[2]*src[4] + tmp[7]*src[5] + tmp[10]*src[7];
	imat[2] -= tmp[3]*src[4] + tmp[6]*src[5] + tmp[11]*src[7];
	imat[3] =  tmp[5]*src[4] + tmp[8]*src[5] + tmp[11]*src[6];
	imat[3] -= tmp[4]*src[4] + tmp[9]*src[5] + tmp[10]*src[6];
	imat[4] =  tmp[1]*src[1] + tmp[2]*src[2] + tmp[5]*src[3];
	imat[4] -= tmp[0]*src[1] + tmp[3]*src[2] + tmp[4]*src[3];
	imat[5] =  tmp[0]*src[0] + tmp[7]*src[2] + tmp[8]*src[3];
	imat[5] -= tmp[1]*src[0] + tmp[6]*src[2] + tmp[9]*src[3];
	imat[6] =  tmp[3]*src[0] + tmp[6]*src[1] + tmp[11]*src[3];
	imat[6] -= tmp[2]*src[0] + tmp[7]*src[1] + tmp[10]*src[3];
	imat[7] =  tmp[4]*src[0] + tmp[9]*src[1] + tmp[10]*src[2];
	imat[7] -= tmp[5]*src[0] + tmp[8]*src[1] + tmp[11]*src[2];

	tmp[0] = src[2]*src[7];
	tmp[1] = src[3]*src[6];
	tmp[2] = src[1]*src[7];
	tmp[3] = src[3]*src[5];
	tmp[4] = src[1]*src[6];
	tmp[5] = src[2]*src[5];
	tmp[6] = src[0]*src[7];
	tmp[7] = src[3]*src[4];
	tmp[8] = src[0]*src[6];
	tmp[9] = src[2]*src[4];
	tmp[10]= src[0]*src[5];
	tmp[11]= src[1]*src[4];

	imat[8] =  tmp[0]*src[13] + tmp[3]*src[14] + tmp[4]*src[15];
	imat[8] -= tmp[1]*src[13] + tmp[2]*src[14] + tmp[5]*src[15];
	imat[9] =  tmp[1]*src[12] + tmp[6]*src[14] + tmp[9]*src[15];
	imat[9] -= tmp[0]*src[12] + tmp[7]*src[14] + tmp[8]*src[15];
	imat[10]=  tmp[2]*src[12] + tmp[7]*src[13] + tmp[10]*src[15];
	imat[10]-= tmp[3]*src[12] + tmp[6]*src[13] + tmp[11]*src[15];
	imat[11]=  tmp[5]*src[12] + tmp[8]*src[13] + tmp[11]*src[14];
	imat[11]-= tmp[4]*src[12] + tmp[9]*src[13] + tmp[10]*src[14];
	imat[12]=  tmp[2]*src[10] + tmp[5]*src[11] + tmp[1]*src[9];
	imat[12]-= tmp[4]*src[11] + tmp[0]*src[9] + tmp[3]*src[10];
	imat[13]=  tmp[8]*src[11] + tmp[0]*src[8] + tmp[7]*src[10];
	imat[13]-= tmp[6]*src[10] + tmp[9]*src[11] + tmp[1]*src[8];
	imat[14]=  tmp[6]*src[9] + tmp[11]*src[11] + tmp[3]*src[8];
	imat[14]-= tmp[10]*src[11] + tmp[2]*src[8] + tmp[7]*src[9];
	imat[15]=  tmp[10]*src[10] + tmp[4]*src[8] + tmp[9]*src[9];
	imat[15]-= tmp[8]*src[9] + tmp[11]*src[10] + tmp[5]*src[8];

	det = src[0]*imat[0] + src[1]*imat[1] + src[2]*imat[2] + src[3]*imat[3];

	if (det != 0.0f)	det = 1.0/det;
	else det = 1.0;
	for (int j=0;j<16;j++)
	{
		imat[j] *= det;
	}
}

#endif